<?php
namespace App\Http\Controllers;

use App\Models\Roles;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;

class RolesController extends Controller
{
    // Create a new role
    public function create(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'description' => 'required|string|max:150',
            'permissions' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Name, description, and permissions are required.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        if (Roles::where('name', $request->name)->whereNull('deleted_at')->exists()) {
            return response()->json(['message' => 'Roles name already exists.'], Response::HTTP_CONFLICT);
        }

        $role = Roles::create($request->all());

        return response()->json(['message' => 'Roles created successfully.', 'data' => $role], Response::HTTP_CREATED);
    }

    // Update a role
    public function update(Request $request, $id)
    {
        $role = Roles::find($id);

        if (!$role || $role->deleted_at) {
            return response()->json(['message' => 'Roles Not Found.'], Response::HTTP_NOT_FOUND);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'description' => 'required|string|max:150',
            'permissions' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Missing required fields.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        $role->update($request->all());

        return response()->json(['message' => 'Roles updated successfully.', 'data' => $role], Response::HTTP_OK);
    }

    // Get all roles
    public function search()
    {
        $roles = Roles::whereNull('deleted_at')->orderBy('created_at', 'desc')->get();
        return response()->json(['message' => 'Roles retrieved successfully.', 'data' => $roles], Response::HTTP_OK);
    }

    // Get a single role by ID
    public function get($id)
    {
        $role = Roles::where('id', $id)->whereNull('deleted_at')->first();

        if (!$role) {
            return response()->json(['message' => 'No role found.'], Response::HTTP_NOT_FOUND);
        }

        return response()->json(['message' => 'Roles retrieved successfully.', 'data' => $role], Response::HTTP_OK);
    }

    // Soft delete a role
    public function destroy($id)
    {
        $role = Roles::find($id);

        if (!$role) {
            return response()->json(['message' => 'Roles not found.'], Response::HTTP_NOT_FOUND);
        }

        $role->deleted_at = now();
        $role->save();

        return response()->json(['message' => 'Roles deleted successfully.', 'data' => $role], Response::HTTP_OK);
    }
}

