<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;

class ContactController extends Controller
{
     public function create(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'firstName' => 'required|string|max:100',
            'lastName' => 'required|string|max:100',
            'email' => 'required|email',
            'phone' => 'nullable|string|max:20',
            'message' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        $contact = Contact::create($request->all());

        return response()->json([
            'message' => 'Contact created successfully.',
            'data' => $contact,
        ], Response::HTTP_CREATED);
    }

    public function update(Request $request, $id)
    {
        $contact = Contact::find($id);
        if (!$contact) {
            return response()->json(['message' => 'Contact not found.'], Response::HTTP_NOT_FOUND);
        }

        $validator = Validator::make($request->all(), [
            'firstName' => 'required|string|max:100',
            'lastName' => 'required|string|max:100',
            'email' => 'required|email|unique:contacts,email,' . $id,
            'phone' => 'nullable|string|max:20',
            'message' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        $contact->update($request->all());

        return response()->json([
            'message' => 'Contact updated successfully.',
            'data' => $contact,
        ], Response::HTTP_OK);
    }

    public function delete($id)
    {
        $contact = Contact::find($id);
        if (!$contact) {
            return response()->json(['message' => 'Contact not found.'], Response::HTTP_NOT_FOUND);
        }

        $contact->delete();

        return response()->json(['message' => 'Contact deleted successfully.'], Response::HTTP_OK);
    }

    public function search(Request $request)
    {
        $searchValue = $request->input('searchValue');
        $query = Contact::orderBy('created_at', 'desc');

        if ($searchValue) {
            $query->where(function ($query) use ($searchValue) {
                $query->where('firstName', 'LIKE', "%{$searchValue}%")
                      ->orWhere('lastName', 'LIKE', "%{$searchValue}%")
                      ->orWhere('email', 'LIKE', "%{$searchValue}%")
                      ->orWhere('phone', 'LIKE', "%{$searchValue}%");
            });
        }

        $contacts = $query->paginate(10);

        return response()->json([
            'message' => 'Contacts retrieved successfully.',
            'data' => $contacts->items(),
            'total' => $contacts->total(),
            'current_page' => $contacts->currentPage(),
            'last_page' => $contacts->lastPage(),
        ], Response::HTTP_OK);
    }
}
